/**
 * Автоматическое обновление БД с IATA данными.
 * Данные разбиваются на 3 таблицы: city, airports и railway.
 * Обновление происходит, если изменился хэш исходных файлов или при первом запуске.
 */
const mysql = require('mysql2/promise');
const axios = require('axios');
const cron = require('node-cron');
const CryptoJS = require('crypto-js');
const fs = require('fs').promises;
const dotenv = require("dotenv"); //Хранение переменных окружения
dotenv.config();

const CITIES_URL = 'https://api.travelpayouts.com/data/ru/cities.json?_gl=1*1mb2jd7*_ga*MTczNzI1MTU1NS4xNzM5MjA0MTQ1*_ga_1WLL0NEBEH*MTc0MjkwNjY4Mi4zLjEuMTc0MjkwNzAxOS42MC4wLjA.';
const AIRPORTS_URL = 'https://api.travelpayouts.com/data/ru/airports.json?_gl=1*eyvlcc*_ga*MTczNzI1MTU1NS4xNzM5MjA0MTQ1*_ga_1WLL0NEBEH*MTc0MjkwNjY4Mi4zLjEuMTc0MjkwNzAxOS42MC4wLjA.';
const HASH_FILE = 'last_hashes.json';

// Функция загрузки данных и вычисления MD5-хэша
async function downloadAndHash(url) {
  console.log(`Начало загрузки данных с URL: ${url}`);
  const response = await axios.get(url, { responseType: 'arraybuffer' });
  if (response.status !== 200) {
    throw new Error(`Ошибка загрузки ${url}: ${response.statusText}`);
  }
  console.log(`Данные успешно загружены с URL: ${url}`);
  const buffer = response.data;
  const hash = CryptoJS.MD5(buffer).toString();
  console.log(`Вычислен MD5-хэш для URL ${url}: ${hash}`);
  const data = JSON.parse(buffer.toString());
  console.log(`JSON успешно распарсен для URL: ${url}`);
  return { data, hash };
}

// Функция загрузки предыдущих хэшей из файла
async function loadLastHashes() {
  console.log('Загрузка предыдущих хэшей...');
  try {
    const content = await fs.readFile(HASH_FILE, 'utf8');
    console.log(`Файл ${HASH_FILE} успешно прочитан.`);
    return JSON.parse(content);
  } catch (err) {
    console.warn(`Файл ${HASH_FILE} не найден. Будет создан новый. Ошибка: ${err.message}`);
    return {};
  }
}

// Функция сохранения новых хэшей в файл
async function saveLastHashes(hashes) {
  console.log(`Сохранение новых хэшей в файл ${HASH_FILE}...`);
  await fs.writeFile(HASH_FILE, JSON.stringify(hashes));
  console.log(`Хэши успешно сохранены в ${HASH_FILE}`);
}

// Функция обновления базы данных: создание 3 таблиц и заполнение данными
async function updateDatabase(citiesData, airportsData, pool) {
    console.log('Начало обновления базы данных...');
    const connection = await pool.getConnection();
    try {
      await connection.beginTransaction();
      console.log('Начата транзакция.');
  
      // --- Загрузка маппингов стран ---
      console.log('Загрузка маппингов стран...');
      const [countryEnRows] = await connection.query('SELECT id, value FROM country_en');
      const [countryRuRows] = await connection.query('SELECT id, value FROM country_ru');
      const countryEnMap = {};
      for (const row of countryEnRows) {
        const key = String(row.id).toLowerCase().trim();
        countryEnMap[key] = row.value ? row.value.trim() : '';
      }
      const countryRuMap = {};
      for (const row of countryRuRows) {
        const key = String(row.id).toLowerCase().trim();
        countryRuMap[key] = row.value ? row.value.trim() : '';
      }
      console.log('Маппинги стран загружены:', countryEnMap, countryRuMap);
  
      // ========= Таблица для городов =========
      console.log('Удаление таблицы city (если существует)...');
      await connection.query('DROP TABLE IF EXISTS city;');
      const createCityTable = `
        CREATE TABLE city (
          id INT AUTO_INCREMENT PRIMARY KEY,
          country_code VARCHAR(2) NOT NULL,
          country_ru VARCHAR(255) NOT NULL,
          country_en VARCHAR(255) NOT NULL,
          city_ru VARCHAR(255),
          city_en VARCHAR(255),
          iata_city CHAR(3),
          time_zone VARCHAR(100),
          lat DECIMAL(9,6),
          lon DECIMAL(9,6)
        );
      `;
      console.log('Создание таблицы city...');
      await connection.query(createCityTable);
      console.log('Таблица city создана.');
  
      // Индексация таблицы city
      const cityIndexes = [
        "CREATE INDEX idx_city_city_ru ON city(city_ru);",
        "CREATE INDEX idx_city_city_en ON city(city_en);",
        "CREATE INDEX idx_city_iata_city ON city(iata_city);"
      ];
      for (const idx of cityIndexes) {
        await connection.query(idx);
      }
      console.log('Индексы для таблицы city созданы.');
  
      // ========= Таблица для аэропортов =========
      console.log('Удаление таблицы airports (если существует)...');
      await connection.query('DROP TABLE IF EXISTS airports;');
      const createAirportsTable = `
        CREATE TABLE airports (
          id INT AUTO_INCREMENT PRIMARY KEY,
          country_code VARCHAR(2) NOT NULL,
          country_ru VARCHAR(255) NOT NULL,
          country_en VARCHAR(255) NOT NULL,
          city_ru VARCHAR(255),
          city_en VARCHAR(255),
          iata_city CHAR(3),
          iata_airport CHAR(3),
          airport_name_ru VARCHAR(255),
          airport_name_en VARCHAR(255),
          time_zone VARCHAR(100),
          lat DECIMAL(9,6),
          lon DECIMAL(9,6)
        );
      `;
      console.log('Создание таблицы airports...');
      await connection.query(createAirportsTable);
      console.log('Таблица airports создана.');
  
      // Индексация таблицы airports
      const airportsIndexes = [
        "CREATE INDEX idx_airports_iata_airport ON airports(iata_airport);",
        "CREATE INDEX idx_airports_airport_name_ru ON airports(airport_name_ru);",
        "CREATE INDEX idx_airports_airport_name_en ON airports(airport_name_en);"
      ];
      for (const idx of airportsIndexes) {
        await connection.query(idx);
      }
      console.log('Индексы для таблицы airports созданы.');
  
      // ========= Таблица для железнодорожных вокзалов =========
      console.log('Удаление таблицы railway (если существует)...');
      await connection.query('DROP TABLE IF EXISTS railway;');
      const createRailwayTable = `
        CREATE TABLE railway (
          id INT AUTO_INCREMENT PRIMARY KEY,
          country_code VARCHAR(2) NOT NULL,
          country_ru VARCHAR(255) NOT NULL,
          country_en VARCHAR(255) NOT NULL,
          city_ru VARCHAR(255),
          city_en VARCHAR(255),
          iata_city CHAR(3),
          iata_railway CHAR(3),
          railway_name_ru VARCHAR(255),
          railway_name_en VARCHAR(255),
          time_zone VARCHAR(100),
          lat DECIMAL(9,6),
          lon DECIMAL(9,6)
        );
      `;
      console.log('Создание таблицы railway...');
      await connection.query(createRailwayTable);
      console.log('Таблица railway создана.');
  
      // Индексация таблицы railway
      const railwayIndexes = [
        "CREATE INDEX idx_railway_iata_railway ON railway(iata_railway);",
        "CREATE INDEX idx_railway_railway_name_ru ON railway(railway_name_ru);",
        "CREATE INDEX idx_railway_railway_name_en ON railway(railway_name_en);"
      ];
      for (const idx of railwayIndexes) {
        await connection.query(idx);
      }
      console.log('Индексы для таблицы railway созданы.');
  
      // ========= Вставка данных в таблицу city =========
      console.log(`Вставка данных городов. Всего городов: ${citiesData.length}`);
      for (const city of citiesData) {
        const cc = (city.country_code || '').toLowerCase().trim();
        const fullCountryRu = countryRuMap[cc] || city.country_code;
        const fullCountryEn = countryEnMap[cc] || city.country_code
        const query = `
          INSERT INTO city 
            (country_code, country_ru, country_en, city_ru, city_en, iata_city, time_zone, lat, lon)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?);
        `;
        await connection.query(query, [
          city.country_code,
          fullCountryRu,
          fullCountryEn,
          city.name,
          city.name_translations?.en || null,
          city.code,
          city.time_zone,
          city.coordinates?.lat,
          city.coordinates?.lon
        ]);
      }
      console.log('Данные городов успешно вставлены.');
  
      // Создаем карту городов по IATA-коду (ключ в нижнем регистре)
      const cityMap = {};
      for (const city of citiesData) {
        if (city.code && city.country_code) {
          cityMap[city.code.toLowerCase()] = city;
        }
      }
      console.log('Карта городов создана.');
  
      // ========= Обработка данных аэропортов =========
      const airportsOnly = airportsData.filter(item => item.iata_type && item.iata_type.toLowerCase() === 'airport');
      const railwayOnly = airportsData.filter(item => item.iata_type && item.iata_type.toLowerCase() === 'railway');
      console.log(`Найдено аэропортов: ${airportsOnly.length}, железнодорожных вокзалов: ${railwayOnly.length}`);
  
      // Вставка данных в таблицу airports
      console.log('Вставка данных аэропортов...');
      for (const airport of airportsOnly) {
        let city_ru = null;
        let city_en = null;
        let iata_city = null;
        if (airport.city_code) {
          const mappedCity = cityMap[airport.city_code.toLowerCase()];
          if (mappedCity) {
            city_ru = mappedCity.name;
            city_en = mappedCity.name_translations?.en || null;
            iata_city = mappedCity.code;
          }
        }
        let airport_name_ru = airport.name;
        if (!airport_name_ru && city_ru) {
          airport_name_ru = city_ru;
        }
        const cc_airport = (airport.country_code || '').toLowerCase().trim();
        const fullCountryRu = countryRuMap[cc_airport] || airport.country_code;
        const fullCountryEn = countryEnMap[cc_airport] || airport.country_code;
        const query = `
          INSERT INTO airports 
            (country_code, country_ru, country_en, city_ru, city_en, iata_city, iata_airport, airport_name_ru, airport_name_en, time_zone, lat, lon)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);
        `;
        await connection.query(query, [
          airport.country_code,
          fullCountryRu,
          fullCountryEn,
          city_ru,
          city_en,
          iata_city,
          airport.code,
          airport_name_ru,
          airport.name_translations?.en || null,
          airport.time_zone,
          airport.coordinates?.lat,
          airport.coordinates?.lon
        ]);
      }
      console.log('Данные аэропортов успешно вставлены.');
  
      // ========= Обработка данных железнодорожных вокзалов =========
      console.log('Вставка данных железнодорожных вокзалов...');
      for (const rail of railwayOnly) {
        let city_ru = null;
        let city_en = null;
        let iata_city = null;
        if (rail.city_code) {
          const mappedCity = cityMap[rail.city_code.toLowerCase()];
          if (mappedCity) {
            city_ru = mappedCity.name;
            city_en = mappedCity.name_translations?.en || null;
            iata_city = mappedCity.code;
          }
        }
        let railway_name_ru = rail.name;
        if (!railway_name_ru && city_ru) {
          railway_name_ru = city_ru;
        }
        const cc_rail = (rail.country_code || '').toLowerCase().trim();
        const fullCountryRuRail = countryRuMap[cc_rail] || rail.country_code;
        const fullCountryEnRail = countryEnMap[cc_rail] || rail.country_code;
        const query = `
          INSERT INTO railway 
            (country_code, country_ru, country_en, city_ru, city_en, iata_city, iata_railway, railway_name_ru, railway_name_en, time_zone, lat, lon)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);
        `;
        await connection.query(query, [
          rail.country_code,
          fullCountryRuRail,
          fullCountryEnRail,
          city_ru,
          city_en,
          iata_city,
          rail.code,
          railway_name_ru,
          rail.name_translations?.en || null,
          rail.time_zone,
          rail.coordinates?.lat,
          rail.coordinates?.lon
        ]);
      }
      console.log('Данные железнодорожных вокзалов успешно вставлены.');
  
      await connection.commit();
      console.log('Транзакция успешно зафиксирована. База данных обновлена.');
    } catch (err) {
      await connection.rollback();
      console.error('Ошибка обновления БД, транзакция отменена:', err);
    } finally {
      connection.release();
      console.log('Соединение с базой данных освобождено.');
    }
  } 

// Основная функция обновления
async function runTask() {
  console.log('----- Начало выполнения задачи обновления -----');
  const pool = mysql.createPool({
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0,
    connectTimeout: 30000
  });

  console.log('Пул подключений создан.');

  const lastHashes = await loadLastHashes();
  console.log('Последние сохраненные хэши:', lastHashes);

  let citiesResult, airportsResult;
  try {
    citiesResult = await downloadAndHash(CITIES_URL);
    airportsResult = await downloadAndHash(AIRPORTS_URL);
  } catch (err) {
    console.error('Ошибка загрузки данных:', err);
    await pool.end();
    return;
  }

  let updateNeeded = false;
  if (lastHashes.cities !== citiesResult.hash) {
    console.log('Обнаружены изменения в данных городов.');
    updateNeeded = true;
  } else {
    console.log('Данные городов не изменились.');
  }
  if (lastHashes.airports !== airportsResult.hash) {
    console.log('Обнаружены изменения в данных аэропортов/railway.');
    updateNeeded = true;
  } else {
    console.log('Данные аэропортов/railway не изменились.');
  }

  if (updateNeeded) {
    console.log('Запуск обновления БД...');
    await updateDatabase(citiesResult.data, airportsResult.data, pool);
    await saveLastHashes({
      cities: citiesResult.hash,
      airports: airportsResult.hash
    });
    console.log('Новые хэши сохранены.');
  } else {
    console.log('Изменений не обнаружено, обновление не требуется.');
  }

  await pool.end();
  console.log('Пул подключений закрыт.');
  console.log('----- Задача обновления завершена -----');
}

// Немедленный запуск для первого заполнения БД
runTask().catch(err => console.error('Ошибка выполнения задачи:', err));

// Планирование обновления по cron – ежедневно в 3:00 по МСК
cron.schedule('0 3 * * *', () => {
  console.log('Запуск задачи cron в:', new Date().toLocaleString('ru-RU', { timeZone: 'Europe/Moscow' }));
  runTask().catch(err => console.error('Ошибка выполнения задачи:', err));
}, {
  timezone: 'Europe/Moscow'
});
