const express = require('express');
const axios = require('axios');
const mysql = require('mysql2/promise');
const cors = require('cors');
const fs = require('fs');
const https = require('https');
const dotenv = require("dotenv"); //Хранение переменных окружения
dotenv.config();

// Настройка HTTPS-сертификатов
const options = {
    key: fs.readFileSync('/home/romyrom/conf/web/ai.kupi.com/ssl/ai.kupi.com.key'),
    cert: fs.readFileSync('/home/romyrom/conf/web/ai.kupi.com/ssl/ai.kupi.com.pem'),
};

const app = express();
app.use(express.json());
//app.use(cors());
const allowedOrigins = [
  "https://ai.kupi.com", // Продакшен
  "http://localhost:3000", // Локальная разработка
  "http://127.0.0.1:3000", // Локальный IP
  "http://192.168.0.108:8080", // Локальный IP для тестов с разных устройств
  "http://169.254.121.56:8080" // Ваш локальный сервер
];

const corsOptions = {
  origin: function (origin, callback) {
      console.log(`CORS: Received request from origin: ${origin}`);
      if (!origin || allowedOrigins.includes(origin)) {
          callback(null, true);
      } else {
          console.warn(`CORS: Blocked request from unauthorized origin: ${origin}`);
          callback(new Error("Not allowed by CORS"));
      }
  },
  credentials: true, // Разрешаем куки
  allowedHeaders: ["Content-Type", "Authorization"], // Разрешаем заголовки
  methods: ["GET", "POST", "PUT", "DELETE", "OPTIONS"], // Разрешаем методы
};

app.use(cors(corsOptions));

// Обрабатываем `OPTIONS` preflight-запросы
app.options("*", cors(corsOptions));

const PORT = 3001;
const server = https.createServer(options, app);

// Настройка пула подключений к БД
const pool = mysql.createPool({
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0,
    connectTimeout: 30000 // 30 секунд таймаут соединения
});

// ====================================================== Маршрут API TAN =================================
// Маршрут для поиска локаций
app.get('/api/v1/search', async (req, res) => {
    try {
      // Получение параметров запроса
      const lang = req.query.lang;
      const queryFragment = req.query.query;
      const apiKey = req.query.apiKey;
      
      // Валидация параметров
      if (!lang || (lang !== 'ru' && lang !== 'en')) {
        return res.status(400).json({ error: 'Параметр lang должен быть "ru" или "en".' });
      }
      if (!queryFragment || queryFragment.length < 3) {
        return res.status(400).json({ error: 'Параметр query должен содержать минимум 3 символа.' });
      }
      if (!apiKey) {
        return res.status(400).json({ error: 'Отсутствует параметр apiKey.' });
      }
      
      console.log(`Запрос поиска. lang=${lang}, query=${queryFragment}, apiKey=${apiKey}`);
      
      // Проверка API-ключа
      const [secretRows] = await pool.query('SELECT * FROM secret WHERE `api-key` = ?', [apiKey]);
      if (!secretRows.length) {
        console.warn(`Неверный API ключ: ${apiKey}`);
        return res.status(403).json({ error: 'Неверный API ключ.' });
      }
      console.log(`API ключ ${apiKey} подтвержден.`);
      
      // Определяем имена столбцов для поиска в зависимости от языка
      const cityNameCol = lang === 'ru' ? 'city_ru' : 'city_en';
      const airportNameCol = lang === 'ru' ? 'airport_name_ru' : 'airport_name_en';
      const railwayNameCol = lang === 'ru' ? 'railway_name_ru' : 'railway_name_en';
      
      // Формирование строки для поиска по префиксу (без учета регистра)
      const fragment = queryFragment.toLowerCase() + '%';
      
      // 1. Поиск прямых совпадений в таблице city
      console.log('Запрос прямых совпадений в таблице city...');
      const [cityRows] = await pool.query(
        `SELECT * FROM city WHERE LOWER(${cityNameCol}) LIKE ? OR LOWER(iata_city) LIKE ?`,
        [fragment, fragment]
      );
      console.log(`Найдено прямых совпадений в city: ${cityRows.length}`);
      
      // Создаем словарь групп по ключу iata_city (в нижнем регистре)
      const groups = {};
      
      // Если найдены записи города, заполняем группы
      cityRows.forEach(city => {
        if (city.iata_city) {
          groups[city.iata_city.toLowerCase()] = {
            city: city,         // Полная запись города
            airports: null,     // null означает, что для этой группы подгрузим все записи
            railway: null
          };
        }
      });
      
      // 2. Поиск совпадений в таблице airports (по названию аэропорта или iata_airport)
      console.log('Запрос совпадений в таблице airports...');
      const [airportMatches] = await pool.query(
        `SELECT * FROM airports WHERE LOWER(${airportNameCol}) LIKE ? OR LOWER(iata_airport) LIKE ?`,
        [fragment, fragment]
      );
      console.log(`Найдено совпадений в airports: ${airportMatches.length}`);
      
      // Группируем совпадения из таблицы airports
      airportMatches.forEach(ap => {
        const key = ap.iata_city ? ap.iata_city.toLowerCase() : 'unknown';
        if (!groups[key]) {
          groups[key] = {
            city: null,       // Записи города нет, создадим заглушку ниже
            airports: [ap],
            railway: []
          };
        } else {
          if (!groups[key].airports) groups[key].airports = [];
          // Проверка на дублирование по id
          if (!groups[key].airports.some(x => x.id === ap.id)) {
            groups[key].airports.push(ap);
          }
        }
      });
      
      // 3. Поиск совпадений в таблице railway (по названию или iata_railway)
      console.log('Запрос совпадений в таблице railway...');
      const [railwayMatches] = await pool.query(
        `SELECT * FROM railway WHERE LOWER(${railwayNameCol}) LIKE ? OR LOWER(iata_railway) LIKE ?`,
        [fragment, fragment]
      );
      console.log(`Найдено совпадений в railway: ${railwayMatches.length}`);
      
      railwayMatches.forEach(rail => {
        const key = rail.iata_city ? rail.iata_city.toLowerCase() : 'unknown';
        if (!groups[key]) {
          groups[key] = {
            city: null,
            airports: [],
            railway: [rail]
          };
        } else {
          if (!groups[key].railway) groups[key].railway = [];
          if (!groups[key].railway.some(x => x.id === rail.id)) {
            groups[key].railway.push(rail);
          }
        }
      });
      
      // 4. Для групп с прямым совпадением по городу (где groups[key].city заполнено)
      // подгружаем все связанные записи из таблиц airports и railway, заменяя ранее найденные частичные совпадения.
      const directCityKeys = Object.keys(groups).filter(key => groups[key].city && groups[key].airports === null);
      if (directCityKeys.length > 0) {
        console.log(`Подгрузка всех аэропортов для прямых совпадений: ${directCityKeys.join(', ')}`);
        const [allAirports] = await pool.query(
          `SELECT * FROM airports WHERE LOWER(iata_city) IN (${directCityKeys.map(() => '?').join(',')})`,
          directCityKeys
        );
        directCityKeys.forEach(key => {
          const deduped = allAirports.filter(ap => ap.iata_city && ap.iata_city.toLowerCase() === key)
            .reduce((acc, curr) => {
              if (!acc.some(x => x.id === curr.id)) acc.push(curr);
              return acc;
            }, []);
          groups[key].airports = deduped;
        });
      }
      
      const directCityKeysRail = Object.keys(groups).filter(key => groups[key].city && groups[key].railway === null);
      if (directCityKeysRail.length > 0) {
        console.log(`Подгрузка всех вокзалов для прямых совпадений: ${directCityKeysRail.join(', ')}`);
        const [allRailway] = await pool.query(
          `SELECT * FROM railway WHERE LOWER(iata_city) IN (${directCityKeysRail.map(() => '?').join(',')})`,
          directCityKeysRail
        );
        directCityKeysRail.forEach(key => {
          const deduped = allRailway.filter(rw => rw.iata_city && rw.iata_city.toLowerCase() === key)
            .reduce((acc, curr) => {
              if (!acc.some(x => x.id === curr.id)) acc.push(curr);
              return acc;
            }, []);
          groups[key].railway = deduped;
        });
      }
      
      // 5. Для групп, где city не заполнено (совпадения найдены только в аэропортах или railway),
      // пытаемся создать "заглушку" по данным совпадения.
      Object.keys(groups).forEach(key => {
        if (!groups[key].city) {
          // Если в группе есть хотя бы один аэропорт, берем данные из него
          if (groups[key].airports && groups[key].airports.length > 0) {
            groups[key].city = {
              type: 'city',
              country_ru: groups[key].airports[0].country_ru,
              country_en: groups[key].airports[0].country_en,
              city_ru: groups[key].airports[0].city_ru,
              city_en: groups[key].airports[0].city_en,
              iata_city: groups[key].airports[0].iata_city,
              time_zone: groups[key].airports[0].time_zone,
              lat: null,
              lon: null
            };
          } else if (groups[key].railway && groups[key].railway.length > 0) {
            groups[key].city = {
              type: 'city',
              country_ru: groups[key].railway[0].country_ru,
              country_en: groups[key].railway[0].country_en,
              city_ru: groups[key].railway[0].city_ru,
              city_en: groups[key].railway[0].city_en,
              iata_city: groups[key].railway[0].iata_city,
              time_zone: groups[key].railway[0].time_zone,
              lat: null,
              lon: null
            };
          }
        }
      });
      
      // Преобразуем словарь групп в массив
      const results = Object.values(groups);
      const total = results.length;
      console.log(`Группировка завершена, итоговых групп: ${total}`);
      
      return res.json({ results, total });
    } catch (err) {
      console.error('Ошибка в маршруте /api/v1/search:', err);
      return res.status(500).json({ error: 'Внутренняя ошибка сервера.' });
    }
  });
//================================================================================ Урезанные данные для форм поиска
  app.get('/api/v1/short', async (req, res) => {
    try {
        // Получение параметров запроса
        const lang = req.query.lang;
        const queryFragment = req.query.query;
        const apiKey = req.query.apiKey;
    
        // Валидация параметров
        if (!lang || (lang !== 'ru' && lang !== 'en')) {
          return res.status(400).json({ error: 'Параметр lang должен быть "ru" или "en".' });
        }
        if (!queryFragment || queryFragment.length < 3) {
          return res.status(400).json({ error: 'Параметр query должен содержать минимум 3 символа.' });
        }
        if (!apiKey) {
          return res.status(400).json({ error: 'Отсутствует параметр apiKey.' });
        }
        console.log(`Запрос поиска простого. lang=${lang}, query=${queryFragment}, apiKey=${apiKey}`);
    
        // Проверка API-ключа
        const [secretRows] = await pool.query('SELECT * FROM secret WHERE `api-key` = ?', [apiKey]);
        if (!secretRows.length) {
          console.warn(`Неверный API ключ: ${apiKey}`);
          return res.status(403).json({ error: 'Неверный API ключ.' });
        }
        console.log(`API ключ ${apiKey} подтвержден.`);
    
        // Определяем столбцы для поиска в зависимости от языка
        const airportNameCol = lang === 'ru' ? 'airport_name_ru' : 'airport_name_en';
        const cityNameCol_ru = 'city_ru'; // всегда используем русские названия для группировки, если они есть
        const cityNameCol_en = 'city_en';
        const countryNameCol_ru = 'country_ru';
        const countryNameCol_en = 'country_en';
    
        // Формируем строку для поиска по префиксу (без учета регистра)
        const fragment = queryFragment.toLowerCase() + '%';
    
        // Выполняем поиск в таблице airports по нескольким столбцам:
        // - По названию аэропорта (выбранному по языку)
        // - По названию города (для обоих языков, т.к. мы хотим найти совпадения и там)
        // - По IATA коду аэропорта и города
        // - По названию страны (на обоих языках)
        let searchQuery;
        let params; // объявляем переменную до использования
        if (lang === 'en') {
        // Для английского языка включаем поиск по IATA кодам
        searchQuery = `
            SELECT * FROM airports
            WHERE 
            LOWER(${airportNameCol}) LIKE ? OR
            LOWER(city_ru) LIKE ? OR
            LOWER(city_en) LIKE ? OR
            LOWER(iata_airport) LIKE ? OR
            LOWER(iata_city) LIKE ? OR
            LOWER(country_ru) LIKE ? OR
            LOWER(country_en) LIKE ?
        `;
        params = [fragment, fragment, fragment, fragment, fragment, fragment, fragment];
        } else {
        // Для русского языка ищем только по названиям (без IATA кода)
        searchQuery = `
            SELECT * FROM airports
            WHERE 
            LOWER(${airportNameCol}) LIKE ? OR
            LOWER(city_ru) LIKE ? OR
            LOWER(city_en) LIKE ? OR
            LOWER(country_ru) LIKE ? OR
            LOWER(country_en) LIKE ?
        `;
        params = [fragment, fragment, fragment, fragment, fragment];
        }
        console.log('Выполняется запрос к таблице airports...');
        const [rows] = await pool.query(searchQuery, params);
        console.log(`Найдено записей в airports: ${rows.length}`);
    
        // Группировка результатов по iata_city (ключ приводим к нижнему регистру)
        const groups = {};
        rows.forEach(record => {
          // Если iata_city отсутствует, используем ключ 'unknown'
          const key = record.iata_city ? record.iata_city.toLowerCase() : 'unknown';
          if (!groups[key]) {
            groups[key] = {
              city: null,
              airports: []
            };
          }
          // Добавляем запись, если еще не добавлена
          if (!groups[key].airports.some(x => x.id === record.id)) {
            groups[key].airports.push(record);
          }
        });
    
        // Формируем объект города для каждой группы.
        // Будем брать данные из первого найденного аэропорта группы.
        function buildCity(record) {
          if (!record) return null;
          return {
            country_code: record.country_code,
            country_ru: record.country_ru,
            country_en: record.country_en,
            city_ru: record.city_ru,
            city_en: record.city_en,
            iata_city: record.iata_city
          };
        }
        // Фильтруем запись аэропорта для ответа – исключаем лишние поля
        function filterAirport(record) {
          return {
            iata_airport: record.iata_airport,
            airport_name_ru: record.airport_name_ru,
            airport_name_en: record.airport_name_en
          };
        }
    
        // Преобразуем группы в итоговый массив без дублирования полей
        const results = Object.keys(groups).map(key => {
          const group = groups[key];
          // Если в группе есть записи, берем город из первого аэропорта
          const cityObj = group.airports.length > 0 ? buildCity(group.airports[0]) : null;
          // Фильтруем список аэропортов
          const airportsList = group.airports.map(filterAirport);
          return {
            city: cityObj,
            airports: airportsList
          };
        });
        const total = results.length;
        console.log(`Группировка завершена, итоговых групп: ${total}`);
    
        return res.json({ results, total });
      } catch (err) {
        console.error('Ошибка в маршруте /api/v1/searchSimple2:', err);
        return res.status(500).json({ error: 'Внутренняя ошибка сервера.' });
      }
  });
  

// ** Запуск сервера **
server.listen(PORT, () => {
    console.log(`Сервер запущен на https://ai.kupi.com:${PORT}`);
});