document.addEventListener('DOMContentLoaded', () => {
    const dateRangeInput = flatpickr("#dateRange", {
      mode: "range",
      dateFormat: "Y-m-d"
    });
  
    const loadLogsBtn = document.getElementById('loadLogs');
    const exportLogsBtn = document.getElementById('exportLogs');
    const sessionsList = document.getElementById('sessionsList');
    const modalOverlay = document.getElementById('modalOverlay');
    const modalClose = document.getElementById('modalClose');
    const sessionDetails = document.getElementById('sessionDetails');
    const sessionInfo = document.getElementById('sessionInfo');
    const tzo = document.getElementById('tzo');
    // Определяем часовой пояс клиента
    const clientTZ = Intl.DateTimeFormat().resolvedOptions().timeZone;
    tzo.innerText = clientTZ;

// 1) Форматируем 14-символьную строку YYYYMMDDHHMMSS
function formatTimestamp(ts) {
    ts = String(ts);
    if (ts.length !== 14) return ts;
    const year   = +ts.slice(0, 4);
    const month  = +ts.slice(4, 6) - 1;
    const day    = +ts.slice(6, 8);
    const hour   = +ts.slice(8, 10);
    const minute = +ts.slice(10, 12);
    const second = +ts.slice(12, 14);
    const date = new Date(Date.UTC(year, month, day, hour, minute, second));
  
    return date.toLocaleString('ru-RU', {
      timeZone: clientTZ,
      day:   '2-digit',
      month: '2-digit',
      year:  'numeric',
      hour:   '2-digit',
      minute: '2-digit'
    });
  }
  
  // 2) Unixtime to Date
  function formatUnixToTimestamp(unixTime) {
    // умножаем на 1000, чтобы из секунд получить миллисекунды
    const date = new Date(unixTime * 1000);
    return date.toLocaleString('ru-RU', {
      day:   '2-digit',
      month: '2-digit',
      year:  'numeric',
      hour:   '2-digit',
      minute: '2-digit'
    });
  } 
      
    loadLogsBtn.onclick = async () => {
      if (dateRangeInput.selectedDates.length !== 2) {
        alert("Выберите диапазон дат");
        return;
      }
  
      const start = dateRangeInput.formatDate(dateRangeInput.selectedDates[0], "Y-m-d");
      const end = dateRangeInput.formatDate(dateRangeInput.selectedDates[1], "Y-m-d");
  
      sessionsList.innerHTML = 'Загрузка...';
      try {
        const res = await fetch(`https://ai.kupi.com:3001/api/logs?start=${start}&end=${end}`);
        const data = await res.json();
  
        if (!Array.isArray(data)) {
          sessionsList.innerHTML = 'Формат ответа некорректен';
          return;
        }
  
        // Создание таблицы
        const table = document.createElement('table');
        table.classList.add('log-table');
  
        const thead = document.createElement('thead');
        thead.innerHTML = `
          <tr>
            <th>№</th>
            <th>IP</th>
            <th>Session ID</th>
            <th>Дата создания</th>
            <th>Дата обновления</th>
          </tr>
        `;
        table.appendChild(thead);
  
        const tbody = document.createElement('tbody');
        data.forEach((session, index) => {
          const row = document.createElement('tr');
          row.classList.add('clickable-row');
          row.onclick = () => openSessionDetails(session.session_id);
          row.innerHTML = `
            <td>${index + 1}</td>
            <td>${session.lastIP || '—'}</td>
            <td>${session.session_id}</td>
            <td>${session.created_at ? formatTimestamp(session.created_at) : '—'}</td>
            <td>${session.last_active ? formatTimestamp(session.last_active) : '—'}</td>
          `;
          tbody.appendChild(row);
        });
        table.appendChild(tbody);
        sessionsList.innerHTML = '';
        sessionsList.appendChild(table);
      } catch (e) {
        sessionsList.innerHTML = 'Ошибка загрузки данных';
        console.error(e);
      }
    };
  
    exportLogsBtn.onclick = () => {
      if (dateRangeInput.selectedDates.length !== 2) {
        alert("Выберите диапазон дат");
        return;
      }
      const start = dateRangeInput.formatDate(dateRangeInput.selectedDates[0], "Y-m-d");
      const end = dateRangeInput.formatDate(dateRangeInput.selectedDates[1], "Y-m-d");
      window.open(`https://ai.kupi.com:3001/api/export_logi?start=${start}&end=${end}`);
    };
  
    modalClose.onclick = () => modalOverlay.classList.add('hidden');
    modalOverlay.onclick = (e) => {
      if (e.target === modalOverlay) modalOverlay.classList.add('hidden');
    };
  
    async function openSessionDetails(sessionId) {
        sessionDetails.innerHTML = 'Загрузка...';
        modalOverlay.classList.remove('hidden');
      
        try {
          const res = await fetch(`https://ai.kupi.com:3001/api/logs/${sessionId}`);
          const session = await res.json();
          const geo = Array.isArray(session.geo) ? session.geo[0] : session.geo;
          const geoCountry = geo?.country || '—';
          const geoCity = geo?.city || '—';
          const geoDate = geo?.date ? formatUnixToTimestamp(geo.date) : '—';
          
          sessionInfo.innerHTML = `
            <strong>Сессия:</strong> ${session.session_id}<br>
            <strong>IP:</strong> ${session.lastIP || 'N/A'}<br>
            <strong>Гео:</strong> ${geoCountry}, ${geoCity}<br>
            <strong>Дата первого входа:</strong> ${geoDate}<br>
            <strong>Последняя активность:</strong> ${formatUnixToTimestamp(session.lastTime)}`;
      
          sessionDetails.innerHTML = '';
          const history = session.history_mess;
      
          if (Array.isArray(history)) {
            history.forEach(entry => {
              const bubble = document.createElement('div');
              bubble.classList.add('message-bubble');
      
              if (entry.who === 'client') {
                bubble.classList.add('user');
              } else if (entry.isSystem) {
                bubble.classList.add('system');
              } else {
                bubble.classList.add('assistant');
              }
      
              const date = new Date(entry.date_unixTime * 1000);
              const timeStr = date.toLocaleString('ru-RU', {
                day: '2-digit', month: '2-digit', year: 'numeric',
                hour: '2-digit', minute: '2-digit'
              });
      
              bubble.innerHTML = `
                <div class="timestamp">${timeStr}</div>
                <div class="meta">
                  <strong>Режим:</strong> ${entry.type || '—'} |
                  <strong>Ответ:</strong> ${entry.err ?? '—'}
                </div>
                <div class="message">${entry.text || '[Пусто]'}</div>
              `;
      
              sessionDetails.appendChild(bubble);
      
              // Обработка ссылок внутри сообщения
              bubble.querySelectorAll('a[href]').forEach(a => {
                try {
                  const decoded = decodeURIComponent(a.href);
                  a.href = decoded;
                  a.textContent = decoded;
                  a.target = "_blank";
                  a.rel = "noopener noreferrer";
      
                  //Кнопка копирования
                  const copyBtn = document.createElement('span');
                  copyBtn.textContent = '📋';
                  copyBtn.title = 'Скопировать ссылку';
                  copyBtn.classList.add('copy-icon');
                  copyBtn.onclick = (e) => {
                    e.preventDefault();
                    const fallbackCopy = (text) => {
                      const textarea = document.createElement('textarea');
                      textarea.value = text;
                      textarea.style.position = 'fixed';
                      textarea.style.opacity = '0';
                      document.body.appendChild(textarea);
                      textarea.focus();
                      textarea.select();
                      try {
                        document.execCommand('copy');
                        copyBtn.textContent = '✅';
                      } catch (err) {
                        console.warn('Не удалось скопировать ссылку через fallback:', err);
                      }
                      document.body.removeChild(textarea);
                      setTimeout(() => (copyBtn.textContent = '📋'), 1500);
                    };
                  
                    const decoded = decodeURIComponent(a.href);
                  
                    if (navigator.clipboard && navigator.clipboard.writeText) {
                      navigator.clipboard.writeText(decoded).then(() => {
                        copyBtn.textContent = '✅';
                        setTimeout(() => (copyBtn.textContent = '📋'), 1500);
                      }).catch(err => {
                        console.warn('Clipboard API не работает, fallback...', err);
                        fallbackCopy(decoded);
                      });
                    } else {
                      fallbackCopy(decoded);
                    }
                  };
                  
      
                  a.after(copyBtn);
                } catch (e) {
                  console.warn('Ошибка декодирования URL:', a.href);
                }
              });
            });
          } else {
            sessionDetails.innerHTML = 'История сообщений отсутствует';
          }
      
        } catch (e) {
          sessionDetails.innerHTML = 'Ошибка загрузки деталей сессии';
          console.error(e);
        }
      }
  });
  