document.addEventListener("DOMContentLoaded", function() {
  
  function getCookie(name) {
    const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
    return match ? match[2] : null;
  }
  //подключение звуковых файлов
  const audioAppearance    = new Audio("https://actions.google.com/sounds/v1/water/air_woosh_underwater.ogg");
  const audioOpenModal     = new Audio("https://actions.google.com/sounds/v1/alarms/beep_short.ogg");
  const audioCloseModal    = new Audio("https://actions.google.com/sounds/v1/cartoon/instrument_strum.ogg");
  const audioAssistantMsg  = new Audio("https://actions.google.com/sounds/v1/cartoon/pop.ogg");
  const audioClientMsg     = new Audio("https://actions.google.com/sounds/v1/cartoon/clang_and_wobble.ogg");
  //const audioVoiceStart    = new Audio("https://actions.google.com/sounds/v1/cartoon/wood_plank_flicks.ogg");
  //const audioVoiceEnd      = new Audio("https://actions.google.com/sounds/v1/impacts/glass_drop_and_roll.ogg");
  //id визуальных элементов
  const mainFrame         = document.getElementById('mainFrame');
  const assistantPanel    = document.getElementById('assistantPanel');
  const modal             = document.getElementById('assistantModal');
  const modalContent      = document.querySelector('.modal-content');
  const closeModalBtn     = document.getElementById('closeModal');
  const radioButtons      = document.getElementsByName('inputMode');
  const chatInputArea     = document.getElementById('chatInputArea');
  const voiceInputArea    = document.getElementById('voiceInputArea');
  const sendButton        = document.getElementById('sendButton');
  const chatInput         = document.getElementById('chatInput');
  const chatHistory       = document.getElementById('chatHistory');
  const toggleChatHistory = document.getElementById('toggleChatHistory');
  const voiceMic          = document.getElementById('voiceMic');
  const voiceReplyCheck   = document.getElementById('voiceReply');
  let lastDateDivider = null;
  let storedSessionId = getCookie('session_id');

  if (voiceReplyCheck) {
    voiceReplyCheck.addEventListener('click', function(e) {
      // отменяем переключение чекбокса
      e.preventDefault();
      // сбрасываем на всякий случай
      this.checked = false;
      // предупреждаем пользователя
      alert('Эта функция сейчас недоступна');
    });
  }

  
  // форматируем дату под формат DD-MText-YYYY
  function formatDateDivider(date) {
    const monthNames = [
      'Января','Февраля','Марта','Апреля','Мая','Июня',
      'Июля','Августа','Сентября','Октября','Ноября','Декабря'
    ];
    const day   = date.getDate();
    const month = monthNames[date.getMonth()];
    const year  = date.getFullYear();
    return `${day} – ${month} – ${year} г.`;
  }

  function typewriterMessage(text, type = 'assistant', speed = 50) {
    const now = new Date();
    const dateKey = now.toDateString();

    if (dateKey !== lastDateDivider) {
      lastDateDivider = dateKey;
      const divider = document.createElement('div');
      divider.classList.add('date-divider');
      divider.textContent = formatDateDivider(now);
      chatHistory.appendChild(divider);
    }
    const bubble = document.createElement('div');
    bubble.classList.add('message', type);
    const label = document.createElement('div');
    label.classList.add('message-label');
    label.textContent = type === 'client' ? 'Вы:' : 'Ассистент:';
    const content = document.createElement('div');
    content.classList.add('message-content');
    const time = document.createElement('div');
    time.classList.add('message-time');
    time.textContent = new Date().toLocaleTimeString().slice(0,5);
    bubble.append(label, content, time);
    chatHistory.appendChild(bubble);
    chatHistory.scrollTop = chatHistory.scrollHeight;
  
    let i = 0;
    const iv = setInterval(() => {
      content.innerHTML = text.slice(0, ++i);
      chatHistory.scrollTop = chatHistory.scrollHeight;
      if (i >= text.length) {
        clearInterval(iv);
        audioAssistantMsg.play().catch(()=>{});
      }
    }, speed);
  }

  // Системное сообщение
  let systemMessageDiv = null;
  function showSystemMessage(text) {
    if (systemMessageDiv) systemMessageDiv.remove();
    systemMessageDiv = document.createElement('div');
    systemMessageDiv.classList.add('message','system');
    systemMessageDiv.innerHTML = `
      <div class="message-label">Система:</div>
      <div class="message-content">${text}</div>
      <div class="message-time">${new Date().toLocaleTimeString().slice(0,5)}</div>`;
    chatHistory.appendChild(systemMessageDiv);
    chatHistory.scrollTop = chatHistory.scrollHeight;
  }
  function clearSystemMessage() {
    if (systemMessageDiv) {
      systemMessageDiv.remove();
      systemMessageDiv = null;
    }
  }

// Новая сессия
function resetChatIfSessionChanged() {
  const currentId = getCookie('session_id');
  if (currentId !== storedSessionId) {
    // сессия сменилась!
    storedSessionId = currentId;
    chatHistory.innerHTML = '';
    lastDateDivider = null;
    handshakeDone = false;
  }
}

  function appendMessage(type, text) {
    const now = new Date();
    const dateKey = now.toDateString();
    if (dateKey !== lastDateDivider) {
      lastDateDivider = dateKey;
      const divider = document.createElement('div');
      divider.classList.add('date-divider');
      divider.textContent = formatDateDivider(now);
      chatHistory.appendChild(divider);
    }
  
    const bubble = document.createElement('div');
    bubble.classList.add('message', type);
    const label = document.createElement('div');
    label.classList.add('message-label');
    label.textContent = type === 'client' ? 'Вы:' : 'Ассистент:';
  
    const content = document.createElement('div');
    content.classList.add('message-content');
  
    if (type === 'assistant') {
      const tmp = document.createElement('div');
      tmp.innerHTML = text;
  
      const links = tmp.querySelectorAll('a[href^="https://www.kupi.com/search?"]');
  
      links.forEach(link => {
        const url = link.href;
  
        // создаём кнопку
        const btn = document.createElement('div');
        btn.textContent = 'Показать найденные билеты';
        btn.className = 'ticket-button';
        btn.style.cursor = 'pointer';
        btn.setAttribute('type', 'button');
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          clearInterval(timer);
          mainFrame.src = url;
          closeModalAnimation();
        });
  
        // создаём алерт
        const alert = document.createElement('div');
        alert.className = 'message-alert';
        let seconds = 12;
        alert.textContent = `Автоматически откроется через 00:${seconds}`;
  
        const timer = setInterval(() => {
          seconds--;
          if (seconds > 9) {
            alert.textContent = `Автоматически откроется через 00:${seconds}`;
          } 
          else if (seconds > 0 && seconds < 10){
            alert.textContent = `Автоматически откроется через 00:0${seconds}`;
          }
          else {
            clearInterval(timer);
            mainFrame.src = url;
            closeModalAnimation();
          }
        }, 1000);
  
        // вставка кнопки и таймера вместо ссылки
        const wrapper = document.createElement('div');
        wrapper.appendChild(btn);
        wrapper.appendChild(alert);
        link.replaceWith(wrapper);
      });
  
      // добавляем содержимое
      content.append(...tmp.childNodes);
    } else {
      content.innerHTML = text;
    }
  
    const time = document.createElement('div');
    time.classList.add('message-time');
    time.textContent = `${String(now.getHours()).padStart(2,'0')}:${String(now.getMinutes()).padStart(2,'0')}`;
  
    bubble.append(label, content, time);
    chatHistory.appendChild(bubble);
    chatHistory.scrollTop = chatHistory.scrollHeight;
  
    if (type === 'client') audioClientMsg.play().catch(() => {});
    else audioAssistantMsg.play().catch(() => {});
  }  
  
  let handshakeDone = false;

  function onFrameFirstLoad() {
    mainFrame.removeEventListener('load', onFrameFirstLoad);
  
    const overlay = document.createElement("div");
    overlay.id = "appearanceOverlay";
    document.body.appendChild(overlay);
    overlay.addEventListener("animationend", () => overlay.remove());
  
    assistantPanel.style.display = "block";
    audioAppearance.play().catch(()=>{});
    void assistantPanel.offsetWidth;
    assistantPanel.classList.add("animate-appear");
  }
  
  mainFrame.addEventListener("load", onFrameFirstLoad);

  assistantPanel.addEventListener("animationend", function(e) {
    if (e.animationName === "widgetMoveDown") {
      const cs = window.getComputedStyle(assistantPanel);
      assistantPanel.style.top = cs.top;
      assistantPanel.style.left = "50%";
      assistantPanel.style.transform = "translate(-50%, 0)";
      assistantPanel.classList.remove("animate-appear");
    }
  });

  // === loadHistory с обязательным showSystemMessage в catch ===
  async function loadHistory() {
    const sessionId = getCookie('session_id');
    if (!sessionId) return;
  
    let timeout = false;
    let timer = null;
  
    try {
      const controller = new AbortController();
      timer = setTimeout(() => {
        timeout = true;
        controller.abort();
      }, 3000);
  
      const res = await fetch(`https://ai.kupi.com:3001/session/${sessionId}/history`, {
        credentials: 'include',
        signal: controller.signal
      });
  
      clearTimeout(timer);
  
      if (res.status === 404) {
        document.cookie = "session_id=; path=/; max-age=0";
        chatHistory.innerHTML = '';
        lastDateDivider = null;
        handshakeDone = false;
        return;
      }
  
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
  
      const { history } = await res.json();
      chatHistory.innerHTML = '';
      lastDateDivider = null;
      clearSystemMessage();
  
      history.forEach(item => {
        const d = new Date(item.date_unixTime * 1000);
        const dateKey = d.toDateString();
  
        if (dateKey !== lastDateDivider) {
          lastDateDivider = dateKey;
          const divider = document.createElement('div');
          divider.classList.add('date-divider');
          divider.textContent = formatDateDivider(d);
          chatHistory.appendChild(divider);
        }
  
        const whoType = item.who === 'assist' ? 'assistant' : 'client';
        const bubble = document.createElement('div');
        bubble.classList.add('message', whoType);
  
        const lbl = document.createElement('div');
        lbl.classList.add('message-label');
        lbl.textContent = item.who === 'assist' ? 'Ассистент:' : 'Вы:';
  
        const cnt = document.createElement('div');
        cnt.classList.add('message-content');
  
        const searchUrlRegex = /<a\s+href="(https:\/\/www\.kupi\.com\/search\?[^"]+)">[^<]*<\/a>/g;
        let raw = item.text;
        let lastIndex = 0;
        let match;
  
        while ((match = searchUrlRegex.exec(raw)) !== null) {
          const url = match[1];
          const before = raw.slice(lastIndex, match.index);
          if (before) {
            const span = document.createElement('span');
            span.innerHTML = before;
            cnt.appendChild(span);
          }
  
          const btn = document.createElement('div');
          btn.textContent = 'Показать найденные билеты';
          btn.className = 'ticket-button his';
          btn.setAttribute('role', 'button');
          btn.style.cursor = 'pointer';
          btn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            setTimeout(() => {
              mainFrame.src = url;
              closeModalAnimation();
            }, 0);
          });
          cnt.appendChild(btn);
  
          lastIndex = match.index + match[0].length;
        }
  
        if (lastIndex < raw.length) {
          const tail = raw.slice(lastIndex);
          if (tail) {
            const span = document.createElement('span');
            span.innerHTML = tail;
            cnt.appendChild(span);
          }
        }
  
        const tm = document.createElement('div');
        tm.classList.add('message-time');
        tm.textContent = `${String(d.getHours()).padStart(2, '0')}:${String(d.getMinutes()).padStart(2, '0')}`;
  
        bubble.append(lbl, cnt, tm);
        chatHistory.appendChild(bubble);
      });
  
      chatHistory.scrollTop = chatHistory.scrollHeight;
  
    } catch (err) {
      clearTimeout(timer);
  
      const isCORS = timeout ||
        err.name === 'AbortError' ||
        (err instanceof TypeError && err.message.includes('Failed to fetch'));
  
      if (isCORS) {
        console.warn('❌ Блокировка CORS или VPN. Ошибка запроса:', err);
  
        if (!chatHistory.children.length) {
          const divider = document.createElement('div');
          divider.classList.add('date-divider');
          divider.textContent = formatDateDivider(new Date());
          chatHistory.appendChild(divider);
        }
  
        const warning = document.createElement('div');
        warning.classList.add('message', 'system');
        warning.innerHTML = `
          <div class="message-label">Система:</div>
          <div class="message-content">Возможно, у вас включен VPN или сервер недоступен. Проверьте соединение.</div>
          <div class="message-time">${new Date().toLocaleTimeString().slice(0,5)}</div>`;
        chatHistory.appendChild(warning);
        chatHistory.scrollTop = chatHistory.scrollHeight;
  
      } else {
        console.warn('Ошибка загрузки истории:', err);
      }
    }
  }

// === openModal принимает mode и сразу переключает input-режим ===
async function openModal(mode) {
  // 0) если сессия сменилась — сбросить историю
  resetChatIfSessionChanged();
  let timer;
  // 1) определяем режим взаимодействия голос/текст
  const interactionMode = mode === 'voice'
    ? (voiceReplyCheck.checked ? 'voice-voice' : 'voice-text')
    : 'text-text';

  // 2) анимация открытия
  audioOpenModal.play().catch(() => {});
  modal.style.display = "block";
  modalContent.style.animation = "modalShow 0.3s ease forwards";
  await new Promise(resolve => {
    modalContent.addEventListener("animationend", function handler(e) {
      if (e.animationName === "modalShow") {
        modalContent.removeEventListener("animationend", handler);
        resolve();
      }
    });
  });

  // 3) переключить интерфейс ввода (чат или голос)
  document.querySelector(`input[name="inputMode"][value="${mode}"]`).checked = true;
  toggleInputArea();

  // 4) если нет куки session_id — это новая сессия, делаем рукопожатие и печатаем приветствие
  const sessionId = getCookie("session_id");
  if (!sessionId) {
    clearSystemMessage();
    try {
      const controller = new AbortController();
      timer = setTimeout(() => controller.abort(), 3000);

      const res = await fetch("https://ai.kupi.com:3001/web_assist", {
        method: "POST",
        credentials: "include",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          input: "",
          interactionMode
        }),
        signal: controller.signal
      });
      clearTimeout(timer);

      const { assistantResponse } = await res.json();
      typewriterMessage(
        assistantResponse?.trim() ||
        "Добро пожаловать! Я ваш персональный ассистент."
      );
    } catch (err) {
      clearTimeout(timer);
      chatHistory.innerHTML = "";
      lastDateDivider = null;
      showSystemMessage(
        "Сервер недоступен или включён VPN. Проверьте соединение и отключите VPN."
      );
    }
    return;
  }

  // 5) иначе — подгрузить уже существующую историю
  await loadHistory();
}

// обработчик клика по иконкам
document.querySelectorAll(".assistant-panel .icon").forEach(icon => {
  icon.addEventListener("click", async () => {
    const mode = icon.dataset.mode;
    await openModal(mode);
    if (mode === "chat") chatInput.focus();
    else voiceMic.focus();
  });
});

  function closeModalAnimation() {
    audioCloseModal.play().catch(()=>{});
    modalContent.style.animation = "modalHide 0.3s ease forwards";
    modalContent.addEventListener("animationend", () => {
      modal.style.display = "none";
      modalContent.style.animation = "modalShow 0.3s ease forwards";
    }, { once: true });
  }

  closeModalBtn.addEventListener('click', closeModalAnimation);
  window.addEventListener('click', e => {
    if (e.target === modal) closeModalAnimation();
  });

  function toggleInputArea() {
    const mode = document.querySelector('input[name="inputMode"]:checked').value;
    if (mode === 'chat') {
      chatInputArea.style.display = 'flex';
      voiceInputArea.style.display = 'none';
    } else {
      chatInputArea.style.display = 'none';
      voiceInputArea.style.display = 'flex';
    }
  }
  radioButtons.forEach(rb => rb.addEventListener('change', toggleInputArea));

  let isDragging = false, offset = { x:0, y:0 };
  function startDrag(e) {
    isDragging = true;
    const x = e.clientX || e.touches[0].clientX;
    const y = e.clientY || e.touches[0].clientY;
    const rect = assistantPanel.getBoundingClientRect();
    offset.x = x - rect.left;
    offset.y = y - rect.top;
  }
  function onDrag(e) {
    if (!isDragging) return;
    const x = e.clientX || e.touches[0].clientX;
    const y = e.clientY || e.touches[0].clientY;
    assistantPanel.style.transform = 'none';
    assistantPanel.style.left = (x - offset.x) + 'px';
    assistantPanel.style.top  = (y - offset.y) + 'px';
    e.preventDefault();
  }
  function endDrag() { isDragging = false; }

  assistantPanel.addEventListener('mousedown', startDrag);
  document.addEventListener('mousemove', onDrag);
  document.addEventListener('mouseup', endDrag);
  assistantPanel.addEventListener('touchstart', startDrag, { passive: false });
  document.addEventListener('touchmove', onDrag, { passive: false });
  document.addEventListener('touchend', endDrag);

// анимация для клиента в ожидании ответа
let processingInterval = null;
let processingStepIndex = 0;
const processingSteps = [
  'Инициализация…',
  'Сбор данных…',
  'Проверка полученных данных…',
  'Запрос к серверу…',
  'Получение ответов…',
  'Анализ вариантов…',
  'Оптимизация маршрута…',
  'Формирование ответа…',
  'Подготовка интерфейса…',
  'Требуется немного больше времени…',
  'Думаю…',
  'Дорабатываю результат…',
  'Корректировка…',
  'Завершение…'
];


function showProcessingWithSteps() {
  // Удаляем, если уже есть
  hideProcessingWithSteps();

  // Создаём контейнер системного сообщения
  const sysBubble = document.createElement('div');
  sysBubble.classList.add('message', 'system');
  sysBubble.id = 'processingBubble';

  const lbl = document.createElement('div');
  lbl.classList.add('message-label');
  lbl.textContent = 'Система:';

  const content = document.createElement('div');
  content.classList.add('message-content');
  content.textContent = 'Ассистент обрабатывает ваш запрос, пожалуйста, подождите...';

  // Контейнер для шагов
  const stepsContainer = document.createElement('div');
  stepsContainer.classList.add('processing-steps');
  content.appendChild(stepsContainer);

  const tm = document.createElement('div');
  tm.classList.add('message-time');
  const now = new Date();
  tm.textContent = `${String(now.getHours()).padStart(2,'0')}:${String(now.getMinutes()).padStart(2,'0')}`;

  sysBubble.append(lbl, content, tm);
  chatHistory.appendChild(sysBubble);
  chatHistory.scrollTop = chatHistory.scrollHeight;

  // запуск анимации шагов
  processingStepIndex = 0;
  stepsContainer.textContent = processingSteps[processingStepIndex];
  processingInterval = setInterval(() => {
    // плавно скрываем
    stepsContainer.classList.add('fade-out');
    setTimeout(() => {
      stepsContainer.classList.remove('fade-out');
      // следующий шаг
      processingStepIndex = (processingStepIndex + 1) % processingSteps.length;
      stepsContainer.textContent = processingSteps[processingStepIndex];
    }, 500); // длительность fade-out
  }, 3300); // интервал между шагами
}

function hideProcessingWithSteps() {
  // Отменяем интервал
  if (processingInterval) {
    clearInterval(processingInterval);
    processingInterval = null;
  }
  // Удаляем пузырь
  const existing = document.getElementById('processingBubble');
  if (existing) existing.remove();
}

// Отправка на сервер
async function sendToServer(text) {
  const mode = document.querySelector('input[name="inputMode"]:checked').value;
  const interactionMode = mode === 'voice'
    ? (voiceReplyCheck.checked ? 'voice-voice' : 'voice-text')
    : 'text-text';

  try {
    // Вывод системного сообщения + шаги
    showProcessingWithSteps();

    const res = await fetch('https://ai.kupi.com:3001/web_assist', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      credentials: 'include',
      body: JSON.stringify({ input: text, interactionMode })
    });
    const data = await res.json();

    // Скрываем всё
    hideProcessingWithSteps();

    // Выводим ответ ассистента
    appendMessage('assistant', data.assistantResponse || 'Возможно я вас не правильно понял. Пожалуйста, повторите или исправьте ваш вопрос.');
  } catch (err) {
    console.error(err);

    // Обязательно скрываем, если ещё показывается
    hideProcessingWithSteps();

    // Выводим сообщение об ошибке
    appendMessage(
      'assistant',
      'Ошибка связи с сервером. Возможно, у вас включен VPN. Проверьте и отключите VPN, если он у вас включен.'
    );
  }
}

  function sendMessage() {
    const txt = chatInput.value.trim();
    if (!txt) return;
    appendMessage('client', txt);
    chatInput.value = '';
    sendToServer(txt);
  }
  sendButton.addEventListener('click', sendMessage);
  chatInput.addEventListener('keydown', e => {
    if (e.key === 'Enter' && e.ctrlKey) sendMessage();
  });

  toggleChatHistory.addEventListener('click', function(e) {
    e.preventDefault();
    if (chatHistory.style.display === 'none') {
      chatHistory.style.display = 'block';
      toggleChatHistory.textContent = 'Скрыть историю';
    } else {
      chatHistory.style.display = 'none';
      toggleChatHistory.textContent = 'Показать историю';
    }
  });

  /*================== АУДИО ВВОД ======================= */
let isRecording = false;       // глобальный флаг
let recorder, chunks = [], recFallback;
let stream, audioCtx, sourceNode, analyser, animationId, silenceTimer, maxTimer;

voiceMic.addEventListener('click', async () => {
  if (isRecording) {
    stopRecording();
    return;
  }

  const isIOS = /iP(hone|od|ad)/.test(navigator.platform)
    || (navigator.platform === 'MacIntel' && navigator.maxTouchPoints > 1);

  const hasStandard = !!(navigator.mediaDevices && navigator.mediaDevices.getUserMedia);
  if (!hasStandard) {
    showSystemMessage('Ваш браузер не поддерживает запись аудио.');
    return;
  }

  try {
    stream = await navigator.mediaDevices.getUserMedia({ audio: true });
  } catch (err) {
    showSystemMessage('Не удалось получить доступ к микрофону.');
    return;
  }

  const AudioCtx = window.AudioContext || window.webkitAudioContext;
  audioCtx = new AudioCtx();
  if (audioCtx.state === 'suspended') await audioCtx.resume();

  sourceNode = audioCtx.createMediaStreamSource(stream);
  analyser = audioCtx.createAnalyser();
  analyser.smoothingTimeConstant = 0.8;
  sourceNode.connect(analyser);
  const dataArray = new Uint8Array(analyser.frequencyBinCount);

  const SILENCE_THRESHOLD = 20;
  silenceTimer = null;

  function animate() {
    analyser.getByteFrequencyData(dataArray);
    const avg = dataArray.reduce((sum, v) => sum + v, 0) / dataArray.length;
    voiceMic.style.transform = `scale(${1 + avg / 300})`;

    if (avg < SILENCE_THRESHOLD && !silenceTimer) {
      silenceTimer = setTimeout(() => stopRecording(), 3000);
    } else if (avg >= SILENCE_THRESHOLD && silenceTimer) {
      clearTimeout(silenceTimer);
      silenceTimer = null;
    }

    animationId = requestAnimationFrame(animate);
  }

  // === Обновляем иконку микрофона на "активную"
  const micImg = voiceMic.querySelector('img');
  if (micImg) {
    micImg.src = 'mic-on.svg';
    micImg.style.width = '30px';
    micImg.style.height = '30px';
  }
  voiceMic.classList.add('recording');
  isRecording = true;
  animate();

  const useMediaRecorder = typeof MediaRecorder !== 'undefined' && !isIOS;
  chunks = [];

  if (useMediaRecorder) {
    recorder = new MediaRecorder(stream, { mimeType: 'audio/webm;codecs=opus' });
    recorder.ondataavailable = e => chunks.push(e.data);
    recorder.start();
  } else {
    recFallback = new Recorder(sourceNode, { numChannels: 1 });
    recFallback.record();
  }

  maxTimer = setTimeout(() => stopRecording(), 30000);
});

// === Общая функция завершения
async function stopRecording() {
  if (!isRecording) return;

  isRecording = false;
  clearTimeout(maxTimer);
  if (silenceTimer) clearTimeout(silenceTimer);
  cancelAnimationFrame(animationId);

  voiceMic.classList.remove('recording');
  voiceMic.style.transform = '';

  const micImg = voiceMic.querySelector('img');
  if (micImg) {
    micImg.src = 'mic.svg';
    micImg.style.width = '';
    micImg.style.height = '';
  }

  if (analyser) analyser.disconnect();
  if (sourceNode) sourceNode.disconnect();
  if (stream) stream.getTracks().forEach(t => t.stop());
  if (audioCtx) await audioCtx.close();

  if (recorder && recorder.state === 'recording') {
    recorder.onstop = async () => {
      const blob = new Blob(chunks, { type: 'audio/webm;codecs=opus' });
      await sendBlob(blob, 'webm');
    };
    recorder.stop();
  } else if (recFallback) {
    recFallback.stop();
    recFallback.exportWAV(async blob => {
      await sendBlob(blob, 'wav');
    });
  }
}

async function sendBlob(blob, format) {
  const form = new FormData();
  form.append('audio', blob, format === 'wav' ? 'recording.wav' : 'recording.webm');
  form.append('interactionMode', voiceReplyCheck.checked ? 'voice-voice' : 'voice-text');
  try {
    const resp = await fetch('https://ai.kupi.com:3001/api/transcribe', {
      method: 'POST',
      body: form
    });
    if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
    const { text } = await resp.json();
    appendMessage('client', text);
    sendToServer(text);
  } catch (err) {
    console.error('Ошибка транскрипции:', err);
    showSystemMessage('Не удалось распознать речь. Попробуйте ещё раз.');
  }
}

});